% Table_4.m
% 
% table of epsilon estimates
% 
%   - have to run epsilon_estimate_bayes first
% 
% "The Past and Future of U.S. Structural Change" 
% Andrew Foerster, Andreas Hornstein, Pierre-Daniel Sarte, Mark Watson
% September 2025
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

% -- Clear Workspace -- %
restoredefaultpath;
clear;
clc;
close all;

% -- File Directories -- % 
datadir = 'Data\DataReplication\DataFinal\'; 
outdir  = 'Tables\';
figdir  = 'Figures\';
matdir  = 'Output\';

% -- Add paths -- %
addpath('Functions\');
addpath('Utilities\')
addpath('Data\')


% -- Load Data and Setup -- %
load_data;
period_lower = 20;       % Shortest Period 
Share_mat = sV_mat;
T = size(Share_mat,1);
% Compute q
q = floor(2*T/period_lower);

load([matdir 'eps_draws_' num2str(q) '.mat']);

% Priors
%  eps
mu_eps_prior = 1.0;
var_eps_prior = (2.0)^2;

% Prior Density
eps_min = 0.0;
eps_max = 10.0;
eps_grid = linspace(eps_min,eps_max,100);
dens_prior_eps = normpdf(eps_grid,mu_eps_prior,sqrt(var_eps_prior));

% Compute density of posteriors
dens_post_eps_m = zeros(length(eps_grid),5);
dens_post_eps_x = zeros(length(eps_grid),5);
% estimate density using non-parametric kernel density
for i = 1:5
    dens_post_eps_m(:,i) = ksdensity(eps_draws_m(:,i),eps_grid,'kernel','normal','width',0.2);
    dens_post_eps_x(:,i) = ksdensity(eps_draws_x(:,i),eps_grid,'kernel','normal','width',0.2);
end

% normalize densities
dens_prior_eps = dens_prior_eps/sum(dens_prior_eps);
dens_post_eps_m = dens_post_eps_m./repmat(sum(dens_post_eps_m),length(eps_grid),1);
dens_post_eps_x = dens_post_eps_x./repmat(sum(dens_post_eps_x),length(eps_grid),1);

% % Create a scatter plot of the data
fig = figure('units','normalized','outerposition',[0 0 0.5 0.95]);
ax_fs = 12;
ax_axis_lab = 12;
for i = 1:5
    subplot(4,3,i)
    plot(eps_grid,dens_prior_eps,'r',eps_grid,dens_post_eps_m(:,i),'b');
    if i == 1
      legend('Prior','Posterior');
    end
    title(['Materials:' SecName(i)],'FontSize',ax_fs);
    xlim([0 8]);
    ax = gca;
    ax.FontSize = ax_fs;
    subplot(4,3,i+5)
    plot(eps_grid,dens_prior_eps,'r',eps_grid,dens_post_eps_x(:,i),'b');
    title(['Investment:' SecName(i)],'FontSize',ax_fs);
    xlim([0 8]);
    ax = gca;
    ax.FontSize = ax_fs;
end
% Save the figure as a png file
figname = [figdir 'est_eps_bayes_' num2str(q) '.png'];
saveas(fig,figname,'png');
close(gcf);

% Compute posterior means and percentiles
pct_post_m = prctile(eps_draws_m,[100*(1/6) 50 100*(5/6)]);
pct_post_x = prctile(eps_draws_x,[100*(1/6) 50 100*(5/6)]);
mean_post_m = mean(eps_draws_m);
mean_post_x = mean(eps_draws_x);

% Save Results
% Save Results
ofile = [outdir 'Table_4.csv'];
fid = fopen(ofile, 'w');

% Save Results
fprintf(fid,'Results for estimating elasticity of substitution using Bayes Methods \n');
% fprintf(fid,'Date: %s\n',this_date);
fprintf('Prior Mean for epsilon: %5.2f\n',mu_eps_prior);
fprintf('Prior Standard Deviation for epsilon: %5.2f\n',sqrt(var_eps_prior));
fprintf(fid,'\n\n');
fprintf(fid,'Results for Material Inputs\n');
fprintf(fid,'Sector,,Posterior Mean,,Posterior Quantiles\n');
fprintf(fid,',,,,16th,50th,83th\n');
for i = 1:5
    tmp = char(SecName(i));
    fprintf(fid,[tmp ',,']);
    fprintf(fid,'%5.2f,,',mean_post_m(i));
    fprintf(fid,'%5.2f,%5.2f,%5.2f \n',pct_post_m(1,i),pct_post_m(2,i),pct_post_m(3,i));
end

fprintf(fid,'\n\nResults for Investment Inputs\n');
for i = 1:5
    tmp = char(SecName(i));
    fprintf(fid,[tmp ',,']);
    fprintf(fid,'%5.2f,,',mean_post_x(i));
    fprintf(fid,'%5.2f,%5.2f,%5.2f \n',pct_post_x(1,i),pct_post_x(2,i),pct_post_x(3,i));
end

fprintf(fid,'\n\n Note: Prior mean and quantiles\n');
% Get draws of epsilon from prior
eps_draws_prior = normrnd(mu_eps_prior,sqrt(var_eps_prior),10000,1);
ii = eps_draws_prior > 0;
eps_draws_prior = eps_draws_prior(ii==1);
pct_prior = prctile(eps_draws_prior,[25 50 75]);
mean_prior = mean(eps_draws_prior);

fprintf(fid,'Prior:,, %5.2f,,',mean_prior);
fprintf(fid,'%5.2f,%5.2f,%5.2f \n',pct_prior(1),pct_prior(2),pct_prior(3));
